options(error = quote({
  dump.frames(to.file=T, dumpto='last.dump')
  load('last.dump.rda')
  print(last.dump)
  q()
}))
######
### Script written by ZTS: 27 Aug 2023 reorganized and automated. 
### following a discussion and presentation by Chris Funk
### The script produces forecast based on NMME SST forecast
### using Pacific indices: Western V gradient, Nino3.4,...
### ZTS: 11 Aug 2022  -- The code only works for nmodel > 1 otherwise the 
###   indexing [,im] will give error. Modify code to work for a single model
######

#library(foreach)
#library(doParallel)
library(RNetCDF)
library(nnet)
library(MASS)
source("templateRPS_Wilks.R")

zproctime0 <- proc.time()
## Set the number of core to be used
# ncores <- detectCores()
#ncores <- 2
#klust <- makeCluster(ncores)
#registerDoParallel(klust)

iMonth <- OMONS  # Change GCM  Initial Month (1==Jan)
SznLen <- SZNLEN   # Number of months is a season
ThisYr <- CURRENTYR
SEASON <- SEASONNAME 
fileprecnc <- OBSPRECIP  #
FirstYr <- FIRSTYR   #1982/2001 depending on data
LastYr  <- LASTYR    #2020    # This is for observations; 
iLead  <- ILEAD   # lead month for the initial month of season
trainLength  <- TRAINLEN # Number of years for training
nprdFirst   <- NPRDFIRST   # Num of prdctors as first param (i.e., #models)
LastYrHind   <- ThisYr   #This is always true. It is model init' year
yrFCST <- ThisYr

gcmStart <- 1    # first index  for regression computation   
gcmLast  <- 6    #length(gcmtype) # last indx of available model

modelFileNames  <- c("canesm5","gem5nemo", "nasa","ccsm4", "cfsv2","cesm1")
forecastNames   <- modelFileNames

DETREND <- FALSE
stdizeObs <- FALSE

#######
PRDCTRS  <- c("IOD","SIOD","Nino34","WVG","PC1","PC2","PC3", "PC4","PC5","PC6")
hnames   <- c("iodhZ","siodhZ","n34hZ","wvGhZ","pc1hZ","pc2hZ","pc3hZ","pc4hZ","pc5hZ","pc6hZ")
fnames   <- c("iodfZ","siodfZ","n34fZ","wvGfZ","pc1fZ","pc2fZ","pc3fZ","pc4fZ","pc5fZ","pc6fZ")
nprdct   <- length(fnames)
VARNMTHD <- "_PRCP_Multinom"
nIndx   <- which(PRDCTRS == "PC1")
varunits = "K"

nmodels <- length(gcmStart:gcmLast)
nmiss    <- 15
rthr     <- 1    # rainfall threshold

if (SznLen == 4) { FTYPES <- "_4monthSeasonal.nc" } else { FTYPES <- "_3monthSeasonal.nc"}

######## Initialize file names and direcotories
dirdat0 <- "./analdat/"
fcstdir <- "./sstdat/"
hinddiro <- "./"
hinddir <- fcstdir

hindcastNames <- c()
eoftsNames <- c()
for (imod in gcmStart:gcmLast) {
     tmp <- paste0(modelFileNames[imod],".nc")
     hindcastNames <- c(hindcastNames, paste0(paste0("sstIndex",SEASON), tmp))
     tmp <- paste0(modelFileNames[imod],".txt")
     eoftsNames <- c(eoftsNames, paste0(paste0("eofts",SEASON), tmp))
}

month.2abb <- c(month.abb, month.abb)

#5######### Get Observational data and its dates

nco <- open.nc(paste0(hinddiro, fileprecnc))
timeo <- var.get.nc(nco, "time")
htimeatt <- att.get.nc(nco, "time", "units")
zodate <- utcal.nc(htimeatt, timeo, type='n') # Works if "days since..."

preco <- var.get.nc(nco, "prec")  #(lon,lat, time) 
lon <- as.character(var.get.nc(nco, "lon"))
lat <- as.character(var.get.nc(nco, "lat"))

######### Use dates to form variable names for indexing
 oDateNames <- paste0(zodate[, 1], SEASON) 
 names(oDateNames) <- oDateNames
 
 dimnames(preco) <- list(lon, lat, oDateNames)
 preco[preco < 0] <- NA

#####
### get years from one of the models. Assume all have the same num yrs
### Code below will determine the right data of obs and models
#### 
 nch <- open.nc(paste0(hinddir, hindcastNames[1]))

 tunitfile <- att.get.nc(nch, "time", "units")

 timeh  <- var.get.nc(nch, "time")
 zhdate <- utcal.nc(tunitfile, timeh, type = "n")

 obsyrs  <- zodate[,1]
 hindyrs <- zhdate[,1]

 lead_to_select <- seq(iLead, iLead+SznLen-1)

 obs_month_to_select <- seq(iMonth + iLead, iMonth + iLead + SznLen - 1)
 tmp <- month.2abb[obs_month_to_select]

 if (SznLen == 4) {
   SEASON1 <- paste0(paste0(paste0(substr(tmp[1],1,1),substr(tmp[2],1,1)),
                            substr(tmp[3],1,1)),
                     substr(tmp[4],1,1))
 } else {
   SEASON1 <- paste0(paste0(substr(tmp[1],1,1),substr(tmp[2],1,1)),
                     substr(tmp[3],1,1))
 }

 if (SEASON != SEASON1) { print(paste("WRONG", SEASON)); break; print(Zewdu)}

 obs_month_to_select <- obs_month_to_select %% 12
 zeroind <- which(obs_month_to_select == 0)
 if (length(zeroind) > 0) obs_month_to_select[zeroind] <- 12

 lastMonth <- obs_month_to_select[length(obs_month_to_select)]
 firstMonth <- obs_month_to_select[1]
 if (lastMonth < (iMonth - 1) && firstMonth < (iMonth - 1)) lastObsYr <- ThisYr else lastObsYr <- ThisYr - 1
 lastMonthIndex <- which(obsyrs == lastObsYr)
 obsyrs <- obsyrs[1:lastMonthIndex]
 
 is_next_year_season = (iMonth + iLead) > 12
 if (is_next_year_season) year_shift <- 1 else year_shift <-0
  
 hind_target_years <- hindyrs + year_shift
 cyrs <- intersect(obsyrs,hind_target_years)

 original_hind_years = cyrs - year_shift
 yrFCST  <- hind_target_years[length(hind_target_years)]
     # This is the actual year the forecast is valid
#####
#### Find the training and testing years for observation and model.
####  They can have different years
#####
 trainYrsObs <- cyrs[1:trainLength]
 testYrsObs  <- cyrs[(trainLength+1):length(cyrs)]
 trainYrsMod  <- original_hind_years[1:trainLength]
 testYrsMod   <- original_hind_years[(trainLength+1):length(cyrs)]
 #NOTE: Length of cyrs is the same as length of original_hind_years

 trainObs  <- paste0(trainYrsObs,SEASON)
 yobs <- preco[,,trainObs]
 testObs <- paste0(testYrsObs,SEASON)
 tOBS  <- preco[,,testObs]

 trainMod <- paste0(trainYrsMod,SEASON)
 testMod  <- paste0(testYrsMod, SEASON)

 fcstyrsMod <- union(testYrsMod, LastYrHind)  # This is test yrs + ThisYr
 fcstyrsObs <- union(testYrsObs,yrFCST)  # This all the forecasts valid for
                                         # including future years if true
 fcstMod  <- paste0(fcstyrsMod,SEASON)
 fcstObs  <- paste0(fcstyrsObs,SEASON)

 allyrsMod <- c(trainMod, fcstMod)
 allyrsObs <- c(trainObs,fcstObs)

#######
### Now prepare data
#######
 yobs <- preco[,,trainObs]
 tOBS  <- preco[,,testObs]

 dimnames(yobs) <- list(lon,lat,trainObs)
 yobs[yobs < rthr ] <- NA
 yobs               <- round(yobs, 1)
 hblw <- apply(yobs,c(1, 2), quantile, probs=1/3, na.rm = TRUE)
 habv <- apply(yobs,c(1, 2), quantile, probs=2/3, na.rm = TRUE)
 ntot <- dim(yobs)[3]  
 hblw  <- round(array(rep(hblw,ntot),c(dim(hblw),ntot)),1)
 habv  <- round(array(rep(habv,ntot),c(dim(habv),ntot)),1)

 oblw <- 1 * (yobs < hblw)
 oabv <- 1 * (yobs > habv)
 onrm <- 1 * (yobs >= hblw & yobs <= habv)

 tOBS[tOBS < rthr ] <- NA
 tOBS  <- round(tOBS, 1)
 tmp1  <- hblw[,,1:dim(tOBS)[3]]
 tmp2  <- habv[,,1:dim(tOBS)[3]]
  
 tblw <- 1 * (tOBS < tmp1)
 tabv <- 1 * (tOBS > tmp2)
 tnrm <- 1 * (tOBS >= tmp1 & tOBS <= tmp2)

                                            #; 3-8N   
 if (stdizeObs)
 {
       # To be consistent with oSdev, need to do mannualy
     y                  <- yobs
     ysum               <- apply(y, c(1,2),sum, na.rm=T)
     ysum[ysum < rthr ] <- NA
     ycnt               <- 1* (!is.na(y))
     ycnt               <- apply(ycnt,c(1,2),sum)  # available data
     ycnt[ycnt <= 5 ]   <- NA
     oMean              <- ysum/ycnt
     oMean              <- array(rep(oMean,ntot),c(dim(oMean),ntot))
     oAnom              <- yobs - oMean

     ycnt               <- ycnt - 1
     anomsum            <- apply(oAnom*oAnom,c(1,2),sum,na.rm=T)
     oSdev              <- sqrt(anomsum/ycnt)
     oSdev[oSdev < 0.1] <- NA
     oSdev              <- array(rep(oSdev,ntot),c(dim(oSdev),ntot))

     oZscore            <- oAnom/oSdev

     hblw <- apply(oZscore,c(1, 2), quantile, probs=1/3, na.rm = TRUE)
     habv <- apply(oZscore,c(1, 2), quantile, probs=2/3, na.rm = TRUE)
     hblw  <- round(array(rep(hblw,ntot),c(dim(hblw),ntot)),1)
     habv  <- round(array(rep(habv,ntot),c(dim(habv),ntot)),1)

     yobs  <- oZscore  # This is the standardied observation

     oblw <- 1 * (yobs < hblw)
     oabv <- 1 * (yobs > habv)
     onrm <- 1 * (yobs >= hblw & yobs <= habv)

     tmp1 <- oMean[,,1:dim(tOBS)[3]]
     tmp2 <- oSdev[,,1:dim(tOBS)[3]]
     tOBS <- (tOBS - tmp1)/tmp2 
     tmp1 <-  hblw[,,1:dim(tOBS)[3]]
     tmp2 <-  habv[,,1:dim(tOBS)[3]]

     tblw <- 1 * (tOBS < tmp1)
     tabv <- 1 * (tOBS > tmp2)
     tnrm <- 1 * (tOBS >= tmp1 & tOBS <= tmp2)

  }

#########################################
#### Get all model data
   wnpAll <- matrix(NA, nrow = length(allyrsMod),
                     ncol = length(forecastNames))
  dimnames(wnpAll) <- list(allyrsMod, forecastNames)

  wepAll <- wnpAll
  ni4All <- wnpAll
  n34All <- wnpAll
  wvgAll <- wnpAll
  iodAll <- wnpAll
  siodAll <- wnpAll
  pc1All  <- wnpAll
  pc2All  <- wnpAll
  pc3All  <- wnpAll
  pc4All  <- wnpAll
  pc5All  <- wnpAll
  pc6All  <- wnpAll


  for (imodel in gcmStart:gcmLast)
  {
     ncf <- open.nc(paste0(fcstdir, hindcastNames[imodel]))
     nch <- open.nc(paste0(hinddir, hindcastNames[imodel]))

####
     tunitfile <- att.get.nc(nch, "time", "units")

     timeh  <- var.get.nc(nch, "time")
     zhdate <- utcal.nc(tunitfile, timeh, type = "n") # works b/se unit is days
     yrs    <- paste0(zhdate[ ,1],SEASON)

     wnp <- var.get.nc(nch, "wnp")  # (lon,lat,time)
     wep <- var.get.nc(nch, "wep")  # (lon,lat,time)
     ni4 <- var.get.nc(nch, "ni4")  # (lon,lat,time)
     n34 <- var.get.nc(nch, "n34")
     wvg <- var.get.nc(nch, "wvg")
     iod <- var.get.nc(nch, "iod")
     siod <- var.get.nc(nch, "siod")
     
     allpc <- read.table(paste0(hinddir, eoftsNames[imodel]))

     wnpAll[yrs,imodel] <- wnp
     wepAll[yrs,imodel] <- wep
     ni4All[yrs,imodel] <- ni4
     n34All[yrs,imodel] <- n34
     wvgAll[yrs,imodel] <- wvg
     iodAll[yrs,imodel] <- iod
     siodAll[yrs,imodel] <- siod
     pc1All[yrs,imodel]  <- allpc[,1]
     pc2All[yrs,imodel]  <- allpc[,2]
     pc3All[yrs,imodel]  <- allpc[,3]
     pc4All[yrs,imodel]  <- allpc[,4]
     pc5All[yrs,imodel]  <- allpc[,5]
     pc6All[yrs,imodel]  <- allpc[,6]

  }

traceback()
##### Done Get models 
 #get selectd yrs which can be different from clim or hindyrs
  wnph <- wnpAll[trainMod,gcmStart:gcmLast]
  weph <- wepAll[trainMod,gcmStart:gcmLast]
  ni4h <- ni4All[trainMod,gcmStart:gcmLast]
  n34h <- n34All[trainMod,gcmStart:gcmLast]
  wvgh <- wvgAll[trainMod,gcmStart:gcmLast]
  iodh <- iodAll[trainMod,gcmStart:gcmLast]
  siodh <- siodAll[trainMod,gcmStart:gcmLast]
  pc1h <- pc1All[trainMod,gcmStart:gcmLast]
  pc2h <- pc2All[trainMod,gcmStart:gcmLast]
  pc3h <- pc3All[trainMod,gcmStart:gcmLast]
  pc4h <- pc4All[trainMod,gcmStart:gcmLast]
  pc5h <- pc5All[trainMod,gcmStart:gcmLast]
  pc6h <- pc6All[trainMod,gcmStart:gcmLast]

  wnpf <- wnpAll[fcstMod,gcmStart:gcmLast]
  wepf <- wepAll[fcstMod,gcmStart:gcmLast]
  ni4f <- ni4All[fcstMod,gcmStart:gcmLast]
  n34f <- n34All[fcstMod,gcmStart:gcmLast]
  wvgf <- wvgAll[fcstMod,gcmStart:gcmLast]
  iodf <- iodAll[fcstMod,gcmStart:gcmLast]
  siodf <- siodAll[fcstMod,gcmStart:gcmLast]
  pc1f <- pc1All[fcstMod,gcmStart:gcmLast]
  pc2f <- pc2All[fcstMod,gcmStart:gcmLast]
  pc3f <- pc3All[fcstMod,gcmStart:gcmLast]
  pc4f <- pc4All[fcstMod,gcmStart:gcmLast]
  pc5f <- pc5All[fcstMod,gcmStart:gcmLast]
  pc6f <- pc6All[fcstMod,gcmStart:gcmLast]

  if (DETREND)
  {
     ones <- seq(1,length(trainMod),1)
     wnphx <- wnph
     wephx <- weph
     ni4hx <- ni4h
     n34hx <- n34h
     wvghx <- wvgh
     iodhx  <- iodh
     siodhx <- siodh
     pc1hx  <- pc1h
     pc2hx  <- pc2h
     pc3hx  <- pc3h
     pc4hx  <- pc4h
     pc5hx  <- pc5h
     pc6hx  <- pc6h

     wnpfx <- wnpf
     wepfx <- wepf
     ni4fx <- ni4f
     n34fx <- n34f
     wvgfx <- wvgf
     iodfx  <- iodf
     siodfx <- siodf
     pc1fx  <- pc1f
     pc2fx  <- pc2f
     pc3fx  <- pc3f
     pc4fx  <- pc4f
     pc5fx  <- pc5f
     pc6fx  <- pc6f

     wnpRgr <- vector("list",length=nmodels)
     wepRgr <- wnpRgr
     ni4Rgr <- wnpRgr
     n34Rgr <- wnpRgr
     wvgRgr <- wnpRgr
     iodRgr <- wnpRgr
     siodRgr <- wnpRgr
     pc1Rgr  <- wnpRgr
     pc2Rgr  <- wnpRgr
     pc3Rgr  <- wnpRgr
     pc4Rgr  <- wnpRgr
     pc5Rgr  <- wnpRgr
     pc6Rgr  <- wnpRgr

     tyr  <- fcstyrsMod  # This includes LastYrHind == ThisYr  
     for (im in 1:nmodels)
     {
#### Western North Pacific
       trnd  <- lm(wnphx[,im] ~ ones)
       vtrnd <- trnd$fitted.values
       xtmp1 <- which(!is.na(vtrnd), arr.ind = TRUE)
       xtmp2 <- which(!is.na(wnphx[,im]), arr.ind = TRUE)
       rtmp  <- intersect(xtmp1, xtmp2)

       wnphx[rtmp,im] <- wnphx[rtmp,im] - vtrnd[rtmp]
       wnpfx[,im]  <- wnpfx[,im] - (trnd$coef[1] + trnd$coef[2] * (tyr - trainYrsMod[1] + 1))
       wnpRgr[[im]] <- trnd

#### Western Equatorial Pacific
       trnd  <- lm(wephx[,im] ~ ones)
       vtrnd <- trnd$fitted.values
       xtmp1 <- which(!is.na(vtrnd), arr.ind = TRUE)
       xtmp2 <- which(!is.na(wephx[,im]), arr.ind = TRUE)
       rtmp  <- intersect(xtmp1, xtmp2)

       wephx[rtmp,im] <- wephx[rtmp,im] - vtrnd[rtmp]
       wepfx[,im]  <- wepfx[,im] - (trnd$coef[1] + trnd$coef[2] * (tyr - trainYrsMod[1] + 1))
       wepRgr[[im]] <- trnd

#### Nino4
       trnd  <- lm(ni4hx[,im] ~ ones)
       vtrnd <- trnd$fitted.values
       xtmp1 <- which(!is.na(vtrnd), arr.ind = TRUE)
       xtmp2 <- which(!is.na(ni4hx[,im]), arr.ind = TRUE)
       rtmp  <- intersect(xtmp1, xtmp2)

       ni4hx[rtmp,im] <- ni4hx[rtmp,im] - vtrnd[rtmp]
       ni4fx[,im]  <- ni4fx[,im] - (trnd$coef[1] + trnd$coef[2] * (tyr - trainYrsMod[1] + 1))
       ni4Rgr[[im]] <- trnd

#### Nino34
       trnd  <- lm(n34hx[,im] ~ ones)
       vtrnd <- trnd$fitted.values
       xtmp1 <- which(!is.na(vtrnd), arr.ind = TRUE)
       xtmp2 <- which(!is.na(n34hx[,im]), arr.ind = TRUE)
       rtmp  <- intersect(xtmp1, xtmp2)

       n34hx[rtmp,im] <- n34hx[rtmp,im] - vtrnd[rtmp]
       n34fx[,im]  <- n34fx[,im] - (trnd$coef[1] + trnd$coef[2] * (tyr - trainYrsMod[1] + 1))
       n34Rgr[[im]] <- trnd

#### Western V Gradient
       trnd  <- lm(wvghx[,im] ~ ones)
       vtrnd <- trnd$fitted.values
       xtmp1 <- which(!is.na(vtrnd), arr.ind = TRUE)
       xtmp2 <- which(!is.na(wvghx[,im]), arr.ind = TRUE)
       rtmp  <- intersect(xtmp1, xtmp2)

       wvghx[rtmp,im] <- wvghx[rtmp,im] - vtrnd[rtmp]
       wvgfx[,im]  <- wvgfx[,im] - (trnd$coef[1] + trnd$coef[2] * (tyr - trainYrsMod[1] + 1))
       wvgRgr[[im]] <- trnd

#### Indian Ocean Dipole (IOD)
       trnd  <- lm(iodhx[,im] ~ ones)
       vtrnd <- trnd$fitted.values
       xtmp1 <- which(!is.na(vtrnd), arr.ind = TRUE)
       xtmp2 <- which(!is.na(iodhx[,im]), arr.ind = TRUE)
       rtmp  <- intersect(xtmp1, xtmp2)

       iodhx[rtmp,im] <- iodhx[rtmp,im] - vtrnd[rtmp]
       iodfx[,im]  <- iodfx[,im] - (trnd$coef[1] + trnd$coef[2] * (tyr - trainYrsMod[1] + 1))
       iodRgr[[im]] <- trnd

#### Southern Indian Ocean Dipole (SIOD)
       trnd  <- lm(siodhx[,im] ~ ones)
       vtrnd <- trnd$fitted.values
       xtmp1 <- which(!is.na(vtrnd), arr.ind = TRUE)
       xtmp2 <- which(!is.na(siodhx[,im]), arr.ind = TRUE)
       rtmp  <- intersect(xtmp1, xtmp2)

       siodhx[rtmp,im] <- siodhx[rtmp,im] - vtrnd[rtmp]
       siodfx[,im]  <- siodfx[,im] - (trnd$coef[1] + trnd$coef[2] * (tyr - trainYrsMod[1] + 1))
       siodRgr[[im]] <- trnd

#### Principal Component 1 of SST
       trnd  <- lm(pc1hx[,im] ~ ones)
       vtrnd <- trnd$fitted.values
       xtmp1 <- which(!is.na(vtrnd), arr.ind = TRUE)
       xtmp2 <- which(!is.na(pc1hx[,im]), arr.ind = TRUE)
       rtmp  <- intersect(xtmp1, xtmp2)

       pc1hx[rtmp,im] <- pc1hx[rtmp,im] - vtrnd[rtmp]
       pc1fx[,im]  <- pc1fx[,im] - (trnd$coef[1] + trnd$coef[2] * (tyr - trainYrsMod[1] + 1))
       pc1Rgr[[im]] <- trnd

#### Principal Component 2 of SST
       trnd  <- lm(pc2hx[,im] ~ ones)
       vtrnd <- trnd$fitted.values
       xtmp1 <- which(!is.na(vtrnd), arr.ind = TRUE)
       xtmp2 <- which(!is.na(pc2hx[,im]), arr.ind = TRUE)
       rtmp  <- intersect(xtmp1, xtmp2)

       pc2hx[rtmp,im] <- pc2hx[rtmp,im] - vtrnd[rtmp]
       pc2fx[,im]  <- pc2fx[,im] - (trnd$coef[1] + trnd$coef[2] * (tyr - trainYrsMod[1] + 1))
       pc2Rgr[[im]] <- trnd


#### Principal Component 3 of SST
       trnd  <- lm(pc3hx[,im] ~ ones)
       vtrnd <- trnd$fitted.values
       xtmp1 <- which(!is.na(vtrnd), arr.ind = TRUE)
       xtmp2 <- which(!is.na(pc3hx[,im]), arr.ind = TRUE)
       rtmp  <- intersect(xtmp1, xtmp2)

       pc3hx[rtmp,im] <- pc3hx[rtmp,im] - vtrnd[rtmp]
       pc3fx[,im]  <- pc3fx[,im] - (trnd$coef[1] + trnd$coef[2] * (tyr - trainYrsMod[1] + 1))
       pc3Rgr[[im]] <- trnd

#### Principal Component 4 of SST
       trnd  <- lm(pc4hx[,im] ~ ones)
       vtrnd <- trnd$fitted.values
       xtmp1 <- which(!is.na(vtrnd), arr.ind = TRUE)
       xtmp2 <- which(!is.na(pc4hx[,im]), arr.ind = TRUE)
       rtmp  <- intersect(xtmp1, xtmp2)

       pc4hx[rtmp,im] <- pc4hx[rtmp,im] - vtrnd[rtmp]
       pc4fx[,im]  <- pc4fx[,im] - (trnd$coef[1] + trnd$coef[2] * (tyr - trainYrsMod[1] + 1))
       pc4Rgr[[im]] <- trnd

#### Principal Component 5 of SST
       trnd  <- lm(pc5hx[,im] ~ ones)
       vtrnd <- trnd$fitted.values
       xtmp1 <- which(!is.na(vtrnd), arr.ind = TRUE)
       xtmp2 <- which(!is.na(pc5hx[,im]), arr.ind = TRUE)
       rtmp  <- intersect(xtmp1, xtmp2)

       pc5hx[rtmp,im] <- pc5hx[rtmp,im] - vtrnd[rtmp]
       pc5fx[,im]  <- pc5fx[,im] - (trnd$coef[1] + trnd$coef[2] * (tyr - trainYrsMod[1] + 1))
       pc5Rgr[[im]] <- trnd
 
#### Principal Component 6 of SST
       trnd  <- lm(pc6hx[,im] ~ ones)
       vtrnd <- trnd$fitted.values
       xtmp1 <- which(!is.na(vtrnd), arr.ind = TRUE)
       xtmp2 <- which(!is.na(pc6hx[,im]), arr.ind = TRUE)
       rtmp  <- intersect(xtmp1, xtmp2)

       pc6hx[rtmp,im] <- pc6hx[rtmp,im] - vtrnd[rtmp]
       pc6fx[,im]  <- pc6fx[,im] - (trnd$coef[1] + trnd$coef[2] * (tyr - trainYrsMod[1] + 1))
       pc6Rgr[[im]] <- trnd

     } # im

     wnph <- wnphx
     weph <- wephx
     ni4h <- ni4hx
     n34h <- n34hx
     wvgh <- wvghx
     iodh <- iodhx
     siodh <- siodhx
     pc1h <- pc1hx
     pc2h <- pc2hx
     pc3h <- pc3hx
     pc4h <- pc4hx
     pc5h <- pc5hx
     pc6h <- pc6hx


     wnpf <- wnpfx
     wepf <- wepfx
     ni4f <- ni4fx
     n34f <- n34fx
     wvgf <- wvgfx
     iodf <- iodfx
     siodf <- siodfx
     pc1f <- pc1fx
     pc2f <- pc2fx
     pc3f <- pc3fx
     pc4f <- pc4fx
     pc5f <- pc5fx
     pc6f <- pc6fx

  } #DETREND

##########################

  wnphZ  <- wnph 
  wephZ  <- weph 
  ni4hZ  <- ni4h
  n34hZ  <- n34h
  wvGhZ  <- wvgh
  iodhZ  <- iodh
  siodhZ <- siodh
  pc1hZ  <- pc1h
  pc2hZ  <- pc2h
  pc3hZ  <- pc3h
  pc4hZ  <- pc4h
  pc5hZ  <- pc5h
  pc6hZ  <- pc6h

  wnpfZ  <- wnpf 
  wepfZ  <- wepf 
  ni4fZ  <- ni4f 
  n34fZ  <- n34f
  wvGfZ  <- wvgf
  iodfZ  <- iodf
  siodfZ <- siodf
  pc1fZ  <- pc1f
  pc2fZ  <- pc2f
  pc3fZ  <- pc3f
  pc4fZ  <- pc4f
  pc5fZ  <- pc5f
  pc6fZ  <- pc6f

#### Can we do for different indexes at once?

 hindcastNames <- hindcastNames[gcmStart:gcmLast]
 for (iprd in 1:nprdFirst)
  {
####### varaibles to store
    for (imodel in 1:nmodels) 
    {
      below <- array(NA,c(length(lon),length(lat)))
      above <- below
      normal <- below
      hrpss  <- below
      trpss  <- below
###### We wanted to use step function to select the best predictors
# for multipredictor logit prediction 
      notiprd <- setdiff(1:nprdct,iprd)
      prdorder <- hnames[notiprd] 
      ssth  <- get(hnames[iprd])
      ssth <- ssth[, imodel]
      xmn <- mean(ssth) 
      xsd <- sd(ssth) 
      if (iprd >= nIndx) { ssth <- (ssth - xmn )/xsd }
       
      sstf  <- get(fnames[iprd])
      sstf  <- sstf[,imodel]
      if (iprd >= nIndx) { sstf <- (sstf - xmn )/xsd }

      for (i in 1:length(notiprd)) { 
        tmp <- get(hnames[notiprd[i]])
        tmp <- tmp[,imodel]
	xmn <- mean(tmp)
	xsd <- sd(tmp)
	if (notiprd[i] >= nIndx) { 
	      ssth <- cbind(ssth,(tmp-xmn)/xsd) 
	    } else    {
              ssth <- cbind(ssth,tmp) 
	}
		    
	tmp <-  get(fnames[notiprd[i]])
	tmp <- tmp[,imodel]
        if (notiprd[i] >= nIndx) {
              sstf <- cbind(sstf,(tmp-xmn)/xsd)
            } else    {
              sstf <- cbind(sstf,tmp)
        }
      }

      tmp <- c(iprd, notiprd)
      dimnames(ssth)[[2]] <- PRDCTRS[tmp]
      dimnames(sstf)[[2]] <- PRDCTRS[tmp]
###########################################
### Do multinominal regression 
####################################### 
      zproctime1 <- proc.time()
      print(paste0("Starting model ",hindcastNames[imodel]))  

      for (ilon in 1:length(lon)) {
        for (jlat in 1:length(lat)) {
           dry   <- oblw[ilon,jlat,]
           wet   <- oabv[ilon,jlat,]
	   nrm   <- onrm[ilon,jlat,]

	   tdry  <- tblw[ilon,jlat,testObs]
	   twet  <- tabv[ilon,jlat,testObs]
	   tavg  <- tnrm[ilon,jlat,testObs]

           prdr   <- ssth
           ytmp <- cbind(dry, prdr)
           xtmp1 <- which(!is.na(ytmp[, 1]), arr.ind = TRUE)
           xtmp2 <- which(!is.na(ytmp[, 2]), arr.ind = TRUE)
           len1 <- length(xtmp1)
           len2 <- length(xtmp2)

           if( len1 >= nmiss & len2 >= nmiss) {
              xtmp <- intersect(xtmp1, xtmp2)
              nlen <- length(xtmp)
              if (nlen > 10) {
	         dry <- dry[xtmp]
		 wet <- wet[xtmp]
		 nrm <- nrm[xtmp]
		 prdr  <- ssth[xtmp,]
		  
                 dataH <- data.frame(dry=dry, normal=nrm, wet= wet,
				     p1=prdr[,1],p2=prdr[,2],p3=prdr[,3], 
				     p4=prdr[,4],p5=prdr[,5],p6=prdr[,6],
				     p7=prdr[,7],p8=prdr[,8],p9=prdr[,9],
				     p10=prdr[,10])
                 dataF <- data.frame(p1=sstf[,1],p2=sstf[,2],p3=sstf[,3],
				     p4=sstf[,4],p5=sstf[,5],p6=sstf[,6],
				     p7=sstf[,7],p8=sstf[,8],p9=sstf[,9],
				     p10=sstf[,10])

		 dataH$category[dataH$dry == 1] <- "dry"
                 dataH$category[dataH$normal == 1] <- "normal"
                 dataH$category[dataH$wet == 1] <- "wet"
                 dataH$category    <- as.factor(dataH$category)
     
                 simlMod <- multinom(category ~ p1, data=dataH, trace=FALSE)
                 fullMod <- multinom(category ~ p1+p2+p3+p4+p5+p6+p7+p8+p9+p10,
                                           data=dataH, trace=FALSE)
                 final_model <- stepAIC(simlMod, scope=list(lower=~p1,
                                        upper=~p1+p2+p3+p4+p5+p6+p7+p8+p9+p10),
                                        direction="forward",trace=0)

                 hcst  <- predict(final_model,dataH,type="probs")
                 tcst  <- predict(final_model,dataF,type="probs")
		 
	         oprob <- dry + 2*nrm + 3*wet
	         hcst <- cbind(oprob, hcst)
	         hrpss[ilon,jlat] <- zrpss(hcst)

                 fcst <- tcst[length(fcstMod),] # forecast valid at future obs
                 tcst <- tcst[testMod,]  #Fcst modl yrs equivalent to testObs

	         oprob <- tdry + 2*tavg + 3*twet
		 
                 tfcst <- cbind(oprob,tcst) # test yrs are < yrFCST
		 id <- is.finite(oprob) & is.finite(apply(tfcst[,2:4],1,sum))
                 tfcst <- tfcst[id,]
              	 #print(tfcst)
		 if(sum(id) >= 3) {
		     trpss[ilon,jlat] <- zrpss(tfcst)
                 } else {
		     trpss[ilon,jlat] <- NA
                 }
                 below[ilon,jlat]  <- 100. * fcst[1]
                 normal[ilon,jlat] <- 100. * fcst[2]
                 above[ilon,jlat]  <- 100. * fcst[3]

              }
           }

        } # ilat
      } # ilon

###################
     dirdat <- paste0(paste0(dirdat0,forecastNames[imodel]),VARNMTHD)
    
     fileName = paste0(PRDCTRS[iprd],FTYPES)
     source("templateWrite2ncMultinomRgr.R")
     print("Done NetCDF for Multinominal Regression")
     tmp <- paste("Completed fcst for imodel = ",imodel)
     tmp <- paste(tmp, " for predictor ")
     print(paste(tmp, iprd))
     print(" ")

     print(date())

     zproctime2 <- proc.time()

     show(zproctime2 - zproctime1)
    
    }  # imodel

  } # iprd  
#stopCluster(klust)

print(date())

zproctime2 <- proc.time()

show(zproctime2 - zproctime0)
